<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Helper functions shared across DevSniper.
 */

/**
 * Safely fetch user meta array.
 *
 * @param int    $user_id User ID.
 * @param string $key     Meta key.
 * @return array
 */
function devsniper_get_user_meta_array( $user_id, $key ) {
	$value = get_user_meta( $user_id, $key, true );
	if ( empty( $value ) || ! is_array( $value ) ) {
		return array();
	}

	return $value;
}

/**
 * Sanitize a string for storage.
 *
 * @param string $value Raw value.
 * @return string
 */
function devsniper_sanitize_string( $value ) {
	return sanitize_text_field( wp_unslash( (string) $value ) );
}

/**
 * Sanitize a URL for storage.
 *
 * @param string $url Raw URL.
 * @return string
 */
function devsniper_sanitize_url( $url ) {
	return esc_url_raw( $url );
}

/**
 * Simple helper to normalize booleans into integers for meta/options.
 *
 * @param mixed $value Any value.
 * @return int
 */
function devsniper_bool_to_int( $value ) {
	return ! empty( $value ) ? 1 : 0;
}

/**
 * Get current post ID on frontend best effort.
 *
 * @return int
 */
function devsniper_current_post_id() {
	$post_id = get_queried_object_id();
	if ( $post_id ) {
		return (int) $post_id;
	}

	global $post;
	if ( $post instanceof WP_Post && $post->ID ) {
		return (int) $post->ID;
	}

	return 0;
}

/**
 * Get current URL (front-end) sanitized.
 *
 * @return string
 */
function devsniper_current_url() {
	$host = isset( $_SERVER['HTTP_HOST'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_HOST'] ) ) : '';
	$uri  = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
	$scheme = is_ssl() ? 'https://' : 'http://';

	if ( $host && $uri ) {
		return esc_url_raw( $scheme . $host . $uri );
	}

	return home_url( '/' );
}

/**
 * Best-effort cache flush for common cache layers.
 *
 * @return void
 */
function devsniper_flush_caches() {
	if ( function_exists( 'wp_cache_flush' ) ) {
		wp_cache_flush();
	}

	if ( function_exists( 'wp_cache_clear_cache' ) ) {
		wp_cache_clear_cache();
	}

	do_action( 'litespeed_purge_all' );
	do_action( 'w3tc_flush_all' );
	do_action( 'rocket_purge_cache' );
	do_action( 'wp_fastest_cache_delete_cache' );
	do_action( 'sg_cachepress_purge_cache' );
	do_action( 'cache_enabler_clear_total_cache' );
}
