<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Front-end Inspector (for admins).
 */
class DevSniper_Inspector {

	/**
	 * Boot hooks.
	 */
	public static function init() {
		add_action( 'wp_enqueue_scripts', array( __CLASS__, 'enqueue_assets' ) );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_elementor_editor_assets' ) );
		add_action( 'wp_footer', array( __CLASS__, 'render_overlay' ) );

		add_action( 'wp_ajax_devsniper_toggle_inspector', array( __CLASS__, 'ajax_toggle_inspector' ) );
		add_action( 'wp_ajax_devsniper_save_element', array( __CLASS__, 'ajax_save_element' ) );
		add_action( 'wp_ajax_devsniper_delete_element', array( __CLASS__, 'ajax_delete_element' ) );
		add_action( 'wp_ajax_devsniper_flush_caches', array( __CLASS__, 'ajax_flush_caches' ) );
	}

	/**
	 * Enqueue inspector assets on frontend for authorized users.
	 */
	public static function enqueue_assets() {
		if ( is_admin() || ! is_user_logged_in() || ! DevSniper_Security::current_user_can_use() ) {
			return;
		}

		wp_enqueue_style(
			'devsniper-inspector',
			DEVSNIPER_URL . 'assets/inspector.css',
			array(),
			DEVSNIPER_VERSION
		);

		wp_enqueue_script(
			'devsniper-inspector',
			DEVSNIPER_URL . 'assets/inspector.js',
			array(),
			DEVSNIPER_VERSION,
			true
		);

		$post_id        = devsniper_current_post_id();
		$edit_post_link = $post_id ? get_edit_post_link( $post_id, 'raw' ) : '';
		$elementor_edit = '';
		if ( $post_id && $edit_post_link ) {
			$elementor_edit = add_query_arg( 'action', 'elementor', $edit_post_link );
		}

		$site_editor_url = '';
		if ( function_exists( 'wp_is_block_theme' ) && wp_is_block_theme() ) {
			$site_editor_url = admin_url( 'site-editor.php' );
		}

		$theme_editor_url = current_user_can( 'edit_themes' ) ? admin_url( 'theme-editor.php' ) : '';
		$custom_css_url   = current_user_can( 'customize' ) ? admin_url( 'customize.php?autofocus[section]=custom_css' ) : '';

		$active_plugins = (array) get_option( 'active_plugins', array() );
		if ( is_multisite() ) {
			$sitewide = (array) get_site_option( 'active_sitewide_plugins', array() );
			if ( $sitewide ) {
				$active_plugins = array_merge( $active_plugins, array_keys( $sitewide ) );
			}
		}
		$active_plugins = array_values( array_unique( $active_plugins ) );

		$all_plugins = array();
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}
		if ( function_exists( 'get_plugins' ) ) {
			$all_plugins = get_plugins();
		}

		$plugin_catalog = array();
		foreach ( $active_plugins as $plugin_file ) {
			$slug = dirname( $plugin_file );
			if ( '.' === $slug ) {
				$slug = basename( $plugin_file, '.php' );
			}
			if ( ! $slug ) {
				continue;
			}
			if ( 'devsniper' === $slug ) {
				continue;
			}
			$name = isset( $all_plugins[ $plugin_file ]['Name'] ) ? $all_plugins[ $plugin_file ]['Name'] : $slug;
			$plugin_catalog[] = array(
				'slug' => $slug,
				'name' => $name,
			);
		}

		$plugin_editor_urls = array();
		foreach ( $active_plugins as $plugin_file ) {
			if ( 0 === strpos( $plugin_file, 'code-snippets/' ) ) {
				$plugin_editor_urls['code-snippets'] = admin_url( 'admin.php?page=snippets' );
			}
			if ( 0 === strpos( $plugin_file, 'code-snippets-pro/' ) ) {
				$plugin_editor_urls['code-snippets-pro'] = admin_url( 'admin.php?page=snippets' );
			}
		}

		wp_localize_script(
			'devsniper-inspector',
			'DevSniperInspectorData',
			array(
				'ajax_url'      => admin_url( 'admin-ajax.php' ),
				'nonce'         => wp_create_nonce( DevSniper_Security::AJAX_NONCE_ACTION ),
				'enabled'       => DevSniper_Favorites::inspector_enabled(),
				'post_id'       => $post_id,
				'edit_post_url' => $edit_post_link,
				'elementor_url' => $elementor_edit,
				'editor_urls'   => array(
					'post'        => $edit_post_link,
					'site_editor' => $site_editor_url,
					'theme_editor'=> $theme_editor_url,
					'custom_css'  => $custom_css_url,
					'plugins'     => $plugin_editor_urls,
				),
				'plugins'       => $plugin_catalog,
				'favorites'     => DevSniper_Favorites::get_element_favorites(),
				'page_url'      => devsniper_current_url(),
				'i18n'          => array(
					'tag'                => esc_html__( 'Tag', 'devsniper' ),
					'id'                 => esc_html__( 'ID', 'devsniper' ),
					'open_editor'        => esc_html__( 'Open editor', 'devsniper' ),
					'copy'               => esc_html__( 'Copy', 'devsniper' ),
					'copied'             => esc_html__( 'Copied!', 'devsniper' ),
					'favorite'           => esc_html__( 'Favorite', 'devsniper' ),
					'favourites'         => esc_html__( 'Favorites', 'devsniper' ),
					'remove'             => esc_html__( 'Remove', 'devsniper' ),
					'no_favorites'       => esc_html__( 'No favorites yet.', 'devsniper' ),
					'search_favorites'   => esc_html__( 'Search favorites…', 'devsniper' ),
					'saved'              => esc_html__( 'Saved', 'devsniper' ),
					'error'              => esc_html__( 'Error', 'devsniper' ),
					'selector'           => esc_html__( 'Selector', 'devsniper' ),
					'path'               => esc_html__( 'Path', 'devsniper' ),
					'source'             => esc_html__( 'Source', 'devsniper' ),
					'size'               => esc_html__( 'Size', 'devsniper' ),
					'classes'            => esc_html__( 'Classes', 'devsniper' ),
					'open'               => esc_html__( 'Open', 'devsniper' ),
					'added_to_favorites' => esc_html__( 'Added to favorites', 'devsniper' ),
					'offsets'            => esc_html__( 'Offsets', 'devsniper' ),
					'close'              => esc_html__( 'Close', 'devsniper' ),
				),
			)
		);
	}

	/**
	 * Enqueue helper script in Elementor editor to auto-select an element passed via URL.
	 *
	 * @param string $hook Current admin hook.
	 */
	public static function enqueue_elementor_editor_assets( $hook ) {
		if ( ! is_admin() || ! is_user_logged_in() || ! DevSniper_Security::current_user_can_use() ) {
			return;
		}

		$action = isset( $_GET['action'] ) ? sanitize_key( wp_unslash( $_GET['action'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( 'elementor' !== $action ) {
			return;
		}

		wp_enqueue_script(
			'devsniper-elementor-select',
			DEVSNIPER_URL . 'assets/elementor-select.js',
			array(),
			DEVSNIPER_VERSION,
			true
		);
	}

	/**
	 * Output overlay containers for inspector.
	 */
	public static function render_overlay() {
		if ( is_admin() || ! is_user_logged_in() || ! DevSniper_Security::current_user_can_use() ) {
			return;
		}
		?>
		<div id="devsniper-overlay" class="devsniper-overlay" style="display:none;"></div>
		<div class="devsniper-inspector-box" style="display:none;"></div>
		<div class="devsniper-inspector-label" style="display:none;"></div>
		<div class="devsniper-inspector-info" style="display:none;"></div>
		<?php
	}

	/**
	 * Toggle inspector enabled state per user.
	 */
	public static function ajax_toggle_inspector() {
		DevSniper_Security::verify_ajax();

		$enabled = isset( $_POST['enabled'] ) ? (bool) $_POST['enabled'] : false; // phpcs:ignore WordPress.Security.NonceVerification.Missing
		DevSniper_Favorites::set_inspector_enabled( $enabled );

		wp_send_json_success(
			array(
				'enabled' => $enabled,
			)
		);
	}

	/**
	 * Save favorite element from inspector.
	 */
	public static function ajax_save_element() {
		DevSniper_Security::verify_ajax();

		$data = array(
			'title'    => isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Missing
			'url'      => isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Missing
			'selector' => isset( $_POST['selector'] ) ? sanitize_text_field( wp_unslash( $_POST['selector'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Missing
			'path'     => isset( $_POST['path'] ) ? sanitize_text_field( wp_unslash( $_POST['path'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Missing
			'source'   => isset( $_POST['source'] ) ? sanitize_text_field( wp_unslash( $_POST['source'] ) ) : '', // phpcs:ignore WordPress.Security.NonceVerification.Missing
		);

		$favs = DevSniper_Favorites::add_element_favorite( $data );

		wp_send_json_success(
			array(
				'favorites' => array_values( $favs ),
			)
		);
	}

	/**
	 * Delete favorite element by ID.
	 */
	public static function ajax_delete_element() {
		DevSniper_Security::verify_ajax();

		$id   = isset( $_POST['id'] ) ? sanitize_text_field( wp_unslash( $_POST['id'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Missing
		$favs = DevSniper_Favorites::remove_element_favorite( $id );

		wp_send_json_success(
			array(
				'favorites' => array_values( $favs ),
			)
		);
	}

	/**
	 * Flush caches after saving updates.
	 */
	public static function ajax_flush_caches() {
		DevSniper_Security::verify_ajax();

		if ( function_exists( 'devsniper_flush_caches' ) ) {
			devsniper_flush_caches();
		}

		wp_send_json_success(
			array(
				'flushed' => true,
			)
		);
	}
}
