<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Manage favourites stored in user meta.
 */
class DevSniper_Favorites {
	const META_SEARCH    = 'devsniper_favorites_search';
	const META_ELEMENTS  = 'devsniper_favorites_elements';
	const META_INSPECTOR = 'devsniper_inspector_enabled';

	/**
	 * Get search favourites for current user.
	 *
	 * @return array
	 */
	public static function get_search_favorites() {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return array();
		}

		return devsniper_get_user_meta_array( $user_id, self::META_SEARCH );
	}

	/**
	 * Toggle search favourite item.
	 *
	 * @param array $item Item data.
	 * @return array Updated favourites.
	 */
	public static function toggle_search_favorite( $item ) {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return array();
		}

		$favs = self::get_search_favorites();
		$id   = isset( $item['id'] ) ? devsniper_sanitize_string( $item['id'] ) : '';

		// Remove existing.
		$favs = array_filter(
			$favs,
			function ( $fav ) use ( $id ) {
				return isset( $fav['id'] ) && $fav['id'] !== $id;
			}
		);

		$add = ! empty( $item['is_active'] );
		if ( $add ) {
			$favs[] = array(
				'id'         => $id,
				'title'      => isset( $item['title'] ) ? devsniper_sanitize_string( $item['title'] ) : '',
				'url'        => isset( $item['url'] ) ? devsniper_sanitize_url( $item['url'] ) : '',
				'thumb'      => isset( $item['thumb'] ) ? devsniper_sanitize_url( $item['thumb'] ) : '',
				'icon_class' => isset( $item['icon_class'] ) ? devsniper_sanitize_string( $item['icon_class'] ) : '',
				'type'       => isset( $item['type'] ) ? devsniper_sanitize_string( $item['type'] ) : '',
				'created_at' => current_time( 'mysql', true ),
			);
		}

		update_user_meta( $user_id, self::META_SEARCH, array_values( $favs ) );

		return $favs;
	}

	/**
	 * Get inspector favourite elements.
	 *
	 * @return array
	 */
	public static function get_element_favorites() {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return array();
		}

		return devsniper_get_user_meta_array( $user_id, self::META_ELEMENTS );
	}

	/**
	 * Save element favourite.
	 *
	 * @param array $data Element data.
	 * @return array Updated favourites.
	 */
	public static function add_element_favorite( $data ) {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return array();
		}

		$favs    = self::get_element_favorites();
		$selector = isset( $data['selector'] ) ? devsniper_sanitize_string( $data['selector'] ) : '';
		$path     = isset( $data['path'] ) ? devsniper_sanitize_string( $data['path'] ) : '';
		$id       = md5( $selector . '|' . $path . '|' . ( isset( $data['url'] ) ? $data['url'] : '' ) );

		// Remove existing with same id.
		$favs = array_filter(
			$favs,
			function ( $fav ) use ( $id ) {
				return isset( $fav['id'] ) && $fav['id'] !== $id;
			}
		);

		$favs[] = array(
			'id'         => $id,
			'title'      => isset( $data['title'] ) ? devsniper_sanitize_string( $data['title'] ) : '',
			'url'        => isset( $data['url'] ) ? devsniper_sanitize_url( $data['url'] ) : '',
			'selector'   => $selector,
			'path'       => $path,
			'source'     => isset( $data['source'] ) ? devsniper_sanitize_string( $data['source'] ) : '',
			'created_at' => current_time( 'mysql', true ),
		);

		update_user_meta( $user_id, self::META_ELEMENTS, array_values( $favs ) );

		return $favs;
	}

	/**
	 * Remove element favourite.
	 *
	 * @param string $id Favourite ID.
	 * @return array
	 */
	public static function remove_element_favorite( $id ) {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return array();
		}

		$favs = self::get_element_favorites();
		$id   = devsniper_sanitize_string( $id );

		$favs = array_filter(
			$favs,
			function ( $fav ) use ( $id ) {
				return isset( $fav['id'] ) && $fav['id'] !== $id;
			}
		);

		update_user_meta( $user_id, self::META_ELEMENTS, array_values( $favs ) );

		return $favs;
	}

	/**
	 * Save inspector enabled flag per user.
	 *
	 * @param bool $enabled Enabled flag.
	 * @return void
	 */
	public static function set_inspector_enabled( $enabled ) {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return;
		}

		update_user_meta( $user_id, self::META_INSPECTOR, devsniper_bool_to_int( $enabled ) );
	}

	/**
	 * Is inspector enabled for current user.
	 *
	 * @return bool
	 */
	public static function inspector_enabled() {
		$user_id = get_current_user_id();
		if ( ! $user_id ) {
			return false;
		}

		$value = get_user_meta( $user_id, self::META_INSPECTOR, true );

		return ! empty( $value );
	}
}
