<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin command palette (search) logic.
 */
class DevSniper_Admin_Search {

	const OPTION_SETTINGS = 'devsniper_settings';
	const OPTION_ADMIN_PAGES = 'devsniper_admin_pages';

	/**
	 * Boot hooks.
	 */
	public static function init() {
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_assets' ) );
		add_action( 'admin_footer', array( __CLASS__, 'render_modal' ) );
		add_action( 'admin_menu', array( __CLASS__, 'register_settings_page' ) );
		add_action( 'admin_menu', array( __CLASS__, 'capture_admin_menu' ), 9999 );

		add_action( 'wp_ajax_devsniper_search', array( __CLASS__, 'ajax_search' ) );
		add_action( 'wp_ajax_devsniper_toggle_favorite', array( __CLASS__, 'ajax_toggle_favorite' ) );
	}

	/**
	 * Plugin settings defaults
	 *
	 * @return array
	 */
	public static function get_settings() {
		$defaults = array(
			'enable_posts'       => 1,
			'enable_users'       => 1,
			'enable_media'       => 1,
			'enable_admin_pages' => 1,
			'enabled'            => 1,
			'keyboard_shortcut'  => 1,
			'shortcut_toggle'    => 1,
		);

		$saved = get_option( self::OPTION_SETTINGS, array() );
		if ( ! is_array( $saved ) ) {
			$saved = array();
		}

		return wp_parse_args( $saved, $defaults );
	}

	/**
	 * Enqueue admin assets.
	 *
	 * @param string $hook Current hook.
	 */
	public static function enqueue_assets( $hook ) {
		if ( ! DevSniper_Security::current_user_can_search() ) {
			return;
		}

		wp_enqueue_style(
			'devsniper-admin-search',
			DEVSNIPER_URL . 'assets/admin-search.css',
			array(),
			DEVSNIPER_VERSION
		);

		wp_enqueue_script(
			'devsniper-admin-search',
			DEVSNIPER_URL . 'assets/admin-search.js',
			array( 'jquery' ),
			DEVSNIPER_VERSION,
			true
		);

		$settings  = self::get_settings();
		$favorites = DevSniper_Favorites::get_search_favorites();

		wp_localize_script(
			'devsniper-admin-search',
			'DevSniperSearchData',
			array(
				'ajax_url'  => admin_url( 'admin-ajax.php' ),
				'nonce'     => wp_create_nonce( DevSniper_Security::AJAX_NONCE_ACTION ),
				'is_admin'  => true,
				'settings'  => array(
					'enabled'            => ! empty( $settings['enabled'] ),
					'enable_posts'       => ! empty( $settings['enable_posts'] ),
					'enable_users'       => ! empty( $settings['enable_users'] ),
					'enable_media'       => ! empty( $settings['enable_media'] ),
					'enable_admin_pages' => ! empty( $settings['enable_admin_pages'] ),
					'keyboard_shortcut'  => ! empty( $settings['keyboard_shortcut'] ),
					'shortcut_toggle'    => ! empty( $settings['shortcut_toggle'] ),
				),
				'favorites' => $favorites,
				'i18n'      => array(
					'placeholder'    => esc_html__( 'Search anything…', 'devsniper' ),
					'no_results'     => esc_html__( 'No results found.', 'devsniper' ),
					'posts_label'    => esc_html__( 'Posts & Pages', 'devsniper' ),
					'users_label'    => esc_html__( 'Users', 'devsniper' ),
					'media_label'    => esc_html__( 'Media', 'devsniper' ),
					'admin_label'    => esc_html__( 'Admin pages', 'devsniper' ),
					'loading'        => esc_html__( 'Searching…', 'devsniper' ),
					'start_typing'   => esc_html__( 'Start typing to search…', 'devsniper' ),
					'error'          => esc_html__( 'Error while searching.', 'devsniper' ),
					'favourites'     => esc_html__( 'Favourites', 'devsniper' ),
					'no_favourites'  => esc_html__( 'No favourites yet – click the star to save a result.', 'devsniper' ),
					'search_fav_tip' => esc_html__( 'Toggle favourite', 'devsniper' ),
				),
			)
		);
	}

	/**
	 * Render modal HTML in admin footer.
	 */
	public static function render_modal() {
		if ( ! DevSniper_Security::current_user_can_search() ) {
			return;
		}
		?>
		<div id="devsniper-overlay" class="devsniper-overlay" style="display:none;"></div>
		<div id="devsniper-modal" class="devsniper-modal" style="display:none;">
			<div class="devsniper-modal-inner">
				<div class="devsniper-modal-header">
					<button type="button" class="devsniper-close" aria-label="<?php esc_attr_e( 'Close', 'devsniper' ); ?>">&times;</button>
					<input type="text" id="devsniper-input" class="devsniper-input" placeholder="<?php echo esc_attr__( 'Search anything… (Ctrl+K)', 'devsniper' ); ?>" autocomplete="off" />
					<button type="button" id="devsniper-favorites-toggle" class="devsniper-favorites-toggle" aria-pressed="false" title="<?php esc_attr_e( 'Show favourite items', 'devsniper' ); ?>">★</button>
				</div>
				<div id="devsniper-results" class="devsniper-results">
					<div class="devsniper-results-empty">
						<?php esc_html_e( 'Start typing to search…', 'devsniper' ); ?>
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Register settings + submenu pages.
	 */
	public static function register_settings_page() {
		if ( ! DevSniper_Security::current_user_can_search() ) {
			return;
		}

		add_menu_page(
			__( 'DevSniper Settings', 'devsniper' ),
			'DevSniper',
			'edit_posts',
			'devsniper',
			array( __CLASS__, 'render_settings_page' ),
			'dashicons-search',
			81
		);

		add_submenu_page(
			'devsniper',
			__( 'DevSniper – General', 'devsniper' ),
			__( 'General', 'devsniper' ),
			'edit_posts',
			'devsniper',
			array( __CLASS__, 'render_settings_page' )
		);

		add_submenu_page(
			'devsniper',
			__( 'DevSniper – Search sources', 'devsniper' ),
			__( 'Search sources', 'devsniper' ),
			'edit_posts',
			'devsniper-sources',
			array( __CLASS__, 'render_settings_page' )
		);

		add_submenu_page(
			'devsniper',
			__( 'DevSniper – Premium', 'devsniper' ),
			__( 'Premium', 'devsniper' ),
			'edit_posts',
			'devsniper-premium',
			array( __CLASS__, 'render_settings_page' )
		);

		add_submenu_page(
			'devsniper',
			__( 'DevSniper – Support', 'devsniper' ),
			__( 'Support', 'devsniper' ),
			'edit_posts',
			'devsniper-support',
			array( __CLASS__, 'render_settings_page' )
		);
	}

	/**
	 * Render settings page with tabs (keeps existing UI).
	 */
	public static function render_settings_page() {
		if ( ! DevSniper_Security::current_user_can_search() ) {
			return;
		}

		$current_tab = 'general';
		if ( isset( $_GET['page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			switch ( sanitize_key( $_GET['page'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
				case 'devsniper-sources':
					$current_tab = 'sources';
					break;
				case 'devsniper-premium':
					$current_tab = 'premium';
					break;
				case 'devsniper-support':
					$current_tab = 'support';
					break;
			}
		}

		if ( isset( $_POST['devsniper_support_submit'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			$current_tab = 'support';
		}

		$support_fields = array(
			'name'    => '',
			'email'   => '',
			'subject' => '',
			'message' => '',
		);
		$support_notice       = '';
		$support_notice_class = '';

		if ( isset( $_POST['devsniper_support_submit'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			check_admin_referer( 'devsniper_support_send', 'devsniper_support_nonce' );

			$support_fields = array(
				'name'    => isset( $_POST['devsniper_support_name'] ) ? sanitize_text_field( wp_unslash( $_POST['devsniper_support_name'] ) ) : '',
				'email'   => isset( $_POST['devsniper_support_email'] ) ? sanitize_email( wp_unslash( $_POST['devsniper_support_email'] ) ) : '',
				'subject' => isset( $_POST['devsniper_support_subject'] ) ? sanitize_text_field( wp_unslash( $_POST['devsniper_support_subject'] ) ) : '',
				'message' => isset( $_POST['devsniper_support_message'] ) ? sanitize_textarea_field( wp_unslash( $_POST['devsniper_support_message'] ) ) : '',
			);

			if ( empty( $support_fields['name'] ) || empty( $support_fields['email'] ) || empty( $support_fields['message'] ) ) {
				$support_notice       = __( 'Please fill in all required fields.', 'devsniper' );
				$support_notice_class = 'notice-error';
			} elseif ( ! is_email( $support_fields['email'] ) ) {
				$support_notice       = __( 'Please enter a valid email address.', 'devsniper' );
				$support_notice_class = 'notice-error';
			} else {
				$subject_line = $support_fields['subject'];
				if ( '' === $subject_line ) {
					$subject_line = __( 'DevSniper Support Request', 'devsniper' );
				}

				$body  = "Name: " . $support_fields['name'] . "\n";
				$body .= "Email: " . $support_fields['email'] . "\n";
				$body .= "Site: " . home_url() . "\n";
				$body .= "User ID: " . get_current_user_id() . "\n\n";
				$body .= "Message:\n" . $support_fields['message'] . "\n";

				$headers = array( 'Content-Type: text/plain; charset=UTF-8' );
				$headers[] = 'Reply-To: ' . $support_fields['name'] . ' <' . $support_fields['email'] . '>';

				$sent = wp_mail( 'mor@interest.org.il', $subject_line, $body, $headers );
				if ( $sent ) {
					$support_notice       = __( 'Thanks! Your message has been sent.', 'devsniper' );
					$support_notice_class = 'notice-success';
					$support_fields['subject'] = '';
					$support_fields['message'] = '';
				} else {
					$support_notice       = __( 'Could not send the message. Please try again.', 'devsniper' );
					$support_notice_class = 'notice-error';
				}
			}
		}

		if ( '' === $support_fields['name'] || '' === $support_fields['email'] ) {
			$current_user = wp_get_current_user();
			if ( '' === $support_fields['name'] ) {
				$support_fields['name'] = $current_user->display_name;
			}
			if ( '' === $support_fields['email'] ) {
				$support_fields['email'] = $current_user->user_email;
			}
		}

		if ( isset( $_POST['devsniper_settings_submit'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			check_admin_referer( 'devsniper_settings_save', 'devsniper_settings_nonce' );

			$settings = array(
				'enabled'            => isset( $_POST['devsniper_enabled'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'keyboard_shortcut'  => isset( $_POST['devsniper_keyboard_shortcut'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'shortcut_toggle'    => isset( $_POST['devsniper_shortcut_toggle'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'enable_posts'       => isset( $_POST['enable_posts'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'enable_users'       => isset( $_POST['enable_users'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'enable_media'       => isset( $_POST['enable_media'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
				'enable_admin_pages' => isset( $_POST['enable_admin_pages'] ) ? 1 : 0, // phpcs:ignore WordPress.Security.NonceVerification.Missing
			);

			update_option( self::OPTION_SETTINGS, $settings, false );
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Settings saved.', 'devsniper' ) . '</p></div>';
		}

		if ( '' !== $support_notice ) {
			echo '<div class="notice ' . esc_attr( $support_notice_class ) . ' is-dismissible"><p>' . esc_html( $support_notice ) . '</p></div>';
		}

		$settings = self::get_settings();
		?>
		<div class="wrap devsniper-settings-wrap">
			<h1>🎯 DevSniper</h1>
			<p><?php esc_html_e( 'Global developer search for your WordPress admin.', 'devsniper' ); ?></p>

			<form method="post">
				<?php wp_nonce_field( 'devsniper_settings_save', 'devsniper_settings_nonce' ); ?>

				<div class="devsniper-settings-tabs">
					<div class="devsniper-settings-tabs-nav">
						<button type="button" class="devsniper-tab-link <?php echo ( 'general' === $current_tab ) ? 'is-active' : ''; ?>" data-tab="devsniper-tab-general">
							<?php esc_html_e( 'General', 'devsniper' ); ?>
						</button>
						<button type="button" class="devsniper-tab-link <?php echo ( 'sources' === $current_tab ) ? 'is-active' : ''; ?>" data-tab="devsniper-tab-sources">
							<?php esc_html_e( 'Search sources', 'devsniper' ); ?>
						</button>
						<button type="button" class="devsniper-tab-link devsniper-tab-link-premium <?php echo ( 'premium' === $current_tab ) ? 'is-active' : ''; ?>" data-tab="devsniper-tab-premium">
							<span class="devsniper-tab-lock" aria-hidden="true">🔒</span>
							<?php esc_html_e( 'Premium', 'devsniper' ); ?>
						</button>
						<button type="button" class="devsniper-tab-link <?php echo ( 'support' === $current_tab ) ? 'is-active' : ''; ?>" data-tab="devsniper-tab-support">
							<?php esc_html_e( 'Support', 'devsniper' ); ?>
						</button>
					</div>

					<div class="devsniper-settings-tabs-panels">
						<div id="devsniper-tab-general" class="devsniper-tab-panel <?php echo ( 'general' === $current_tab ) ? 'is-active' : ''; ?>">
							<h2><?php esc_html_e( 'General settings', 'devsniper' ); ?></h2>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><?php esc_html_e( 'Enable DevSniper', 'devsniper' ); ?></th>
										<td>
											<label><input type="checkbox" name="devsniper_enabled" value="1" <?php checked( $settings['enabled'], 1 ); ?> /> <?php esc_html_e( 'Turn DevSniper on or off without deactivating the plugin.', 'devsniper' ); ?></label>
										</td>
									</tr>
									<tr>
										<th scope="row"><?php esc_html_e( 'Keyboard shortcut', 'devsniper' ); ?></th>
										<td>
											<label><input type="checkbox" name="devsniper_keyboard_shortcut" value="1" <?php checked( $settings['keyboard_shortcut'], 1 ); ?> /> <?php esc_html_e( 'Enable the Ctrl+K (Windows) / Cmd+K (macOS) shortcut to open DevSniper.', 'devsniper' ); ?></label>
										</td>
									</tr>
									<tr>
										<th scope="row"><?php esc_html_e( 'Shortcut behaviour', 'devsniper' ); ?></th>
										<td>
											<label><input type="checkbox" name="devsniper_shortcut_toggle" value="1" <?php checked( $settings['shortcut_toggle'], 1 ); ?> /> <?php esc_html_e( 'When DevSniper is open, press the shortcut again to close it.', 'devsniper' ); ?></label>
											<p class="description"><?php esc_html_e( 'Recommended for developers who use the command palette all day long.', 'devsniper' ); ?></p>
										</td>
									</tr>
								</tbody>
							</table>
						</div>

						<div id="devsniper-tab-sources" class="devsniper-tab-panel <?php echo ( 'sources' === $current_tab ) ? 'is-active' : ''; ?>">
							<h2><?php esc_html_e( 'Search sources', 'devsniper' ); ?></h2>
							<table class="form-table" role="presentation">
								<tbody>
									<tr>
										<th scope="row"><?php esc_html_e( 'Posts & Pages', 'devsniper' ); ?></th>
										<td><label><input type="checkbox" name="enable_posts" value="1" <?php checked( $settings['enable_posts'], 1 ); ?> /> <?php esc_html_e( 'Include posts and pages in search results.', 'devsniper' ); ?></label></td>
									</tr>
									<tr>
										<th scope="row"><?php esc_html_e( 'Users', 'devsniper' ); ?></th>
										<td><label><input type="checkbox" name="enable_users" value="1" <?php checked( $settings['enable_users'], 1 ); ?> /> <?php esc_html_e( 'Include users in search results.', 'devsniper' ); ?></label></td>
									</tr>
									<tr>
										<th scope="row"><?php esc_html_e( 'Media', 'devsniper' ); ?></th>
										<td><label><input type="checkbox" name="enable_media" value="1" <?php checked( $settings['enable_media'], 1 ); ?> /> <?php esc_html_e( 'Include media library items in search results.', 'devsniper' ); ?></label></td>
									</tr>
									<tr>
										<th scope="row"><?php esc_html_e( 'Admin pages', 'devsniper' ); ?></th>
										<td><label><input type="checkbox" name="enable_admin_pages" value="1" <?php checked( $settings['enable_admin_pages'], 1 ); ?> /> <?php esc_html_e( 'Include admin menu pages (plugins, settings, tools, etc.).', 'devsniper' ); ?></label></td>
									</tr>
								</tbody>
							</table>
						</div>

						<div id="devsniper-tab-premium" class="devsniper-tab-panel <?php echo ( 'premium' === $current_tab ) ? 'is-active' : ''; ?>">
							<h2><?php esc_html_e( 'DevSniper Premium', 'devsniper' ); ?> <span class="devsniper-premium-badge"><?php esc_html_e( 'Coming soon', 'devsniper' ); ?></span></h2>
							<p class="devsniper-premium-intro"><?php esc_html_e( 'Compare Free vs Premium and see what is coming next for teams and power users.', 'devsniper' ); ?></p>
							<div class="devsniper-premium-compare" aria-label="<?php esc_attr_e( 'DevSniper plan comparison', 'devsniper' ); ?>">
								<table class="devsniper-premium-compare-table" role="table">
									<thead>
										<tr>
											<th scope="col"><?php esc_html_e( 'Feature', 'devsniper' ); ?></th>
											<th scope="col" class="devsniper-compare-free"><?php esc_html_e( 'Free', 'devsniper' ); ?></th>
											<th scope="col" class="devsniper-compare-pro"><?php esc_html_e( 'Premium', 'devsniper' ); ?></th>
										</tr>
									</thead>
									<tbody>
										<tr>
											<th scope="row"><?php esc_html_e( 'Admin command palette search', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Keyboard shortcut (Ctrl/Cmd+K)', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Search posts and pages', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Search users', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Search media library', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Search admin menu pages', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Per-user favorites', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Basic settings toggles', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'No tracking or external requests', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-check">V</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr class="devsniper-compare-highlight">
											<th scope="row"><?php esc_html_e( 'Live text and color editing', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Custom themes and branding presets', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Advanced filters for CPTs, taxonomies, and meta', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Team shared favorites and presets', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Role-based access and source rules', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Performance boost for large sites', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Search analytics and audit log', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Premium support with fast response', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Early access updates and new features', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'Premium onboarding and setup help', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
										<tr>
											<th scope="row"><?php esc_html_e( 'White-label footer and branding control', 'devsniper' ); ?></th>
											<td class="devsniper-compare-free"><span class="devsniper-compare-cross">X</span></td>
											<td class="devsniper-compare-pro"><span class="devsniper-compare-check">V</span></td>
										</tr>
									</tbody>
								</table>
							</div>
							<div class="devsniper-premium-cta-row">
								<a class="devsniper-premium-cta" href="https://interest.org.il" target="_blank" rel="noopener noreferrer">
									<?php esc_html_e( 'Upgrade now', 'devsniper' ); ?>
								</a>
							</div>
							<p class="devsniper-premium-footer"><?php esc_html_e( 'Premium is actively evolving - more features, improvements, and integrations will be added over time.', 'devsniper' ); ?></p>
						</div>

						<div id="devsniper-tab-support" class="devsniper-tab-panel <?php echo ( 'support' === $current_tab ) ? 'is-active' : ''; ?>">
							<h2><?php esc_html_e( 'Support', 'devsniper' ); ?></h2>
							<p class="devsniper-support-intro"><?php esc_html_e( 'Need help or want to request a feature? Send us a message and we will reply shortly.', 'devsniper' ); ?></p>
							<div class="devsniper-support-card">
								<div class="devsniper-support-card-header">
									<span class="devsniper-support-badge"><?php esc_html_e( 'Priority for Premium users', 'devsniper' ); ?></span>
									<p class="devsniper-support-note"><?php esc_html_e( 'Typical response: within 24 hours on business days.', 'devsniper' ); ?></p>
								</div>
								<div class="devsniper-support-grid">
									<div class="devsniper-support-field">
										<label for="devsniper-support-name"><?php esc_html_e( 'Full name', 'devsniper' ); ?> *</label>
										<input type="text" id="devsniper-support-name" name="devsniper_support_name" value="<?php echo esc_attr( $support_fields['name'] ); ?>" required />
									</div>
									<div class="devsniper-support-field">
										<label for="devsniper-support-email"><?php esc_html_e( 'Email', 'devsniper' ); ?> *</label>
										<input type="email" id="devsniper-support-email" name="devsniper_support_email" value="<?php echo esc_attr( $support_fields['email'] ); ?>" required />
									</div>
									<div class="devsniper-support-field devsniper-support-field-full">
										<label for="devsniper-support-subject"><?php esc_html_e( 'Subject', 'devsniper' ); ?></label>
										<input type="text" id="devsniper-support-subject" name="devsniper_support_subject" value="<?php echo esc_attr( $support_fields['subject'] ); ?>" placeholder="<?php esc_attr_e( 'Support request', 'devsniper' ); ?>" />
									</div>
									<div class="devsniper-support-field devsniper-support-field-full">
										<label for="devsniper-support-message"><?php esc_html_e( 'Message', 'devsniper' ); ?> *</label>
										<textarea id="devsniper-support-message" name="devsniper_support_message" rows="5" required><?php echo esc_textarea( $support_fields['message'] ); ?></textarea>
									</div>
								</div>
								<div class="devsniper-support-actions">
									<?php wp_nonce_field( 'devsniper_support_send', 'devsniper_support_nonce' ); ?>
									<button type="submit" name="devsniper_support_submit" class="button button-primary devsniper-support-submit"><?php esc_html_e( 'Send support request', 'devsniper' ); ?></button>
								</div>
							</div>
						</div>
					</div>
				</div>

				<?php submit_button( __( 'Save settings', 'devsniper' ), 'primary', 'devsniper_settings_submit' ); ?>
			</form>

			<hr/>
			<h2><?php esc_html_e( 'How to use', 'devsniper' ); ?></h2>
			<ul class="devsniper-howto-list">
				<li><?php esc_html_e( 'From any screen in wp-admin, press Ctrl+K (Windows) or Cmd+K (macOS).', 'devsniper' ); ?></li>
				<li><?php esc_html_e( 'Start typing to search posts, pages, media, users and admin pages.', 'devsniper' ); ?></li>
				<li><?php esc_html_e( 'Use the arrow keys to move between results and Enter to open.', 'devsniper' ); ?></li>
			</ul>
			<p class="devsniper-settings-footer">DevSniper &copy; <?php echo esc_html( gmdate( 'Y' ) ); ?> &nbsp;|&nbsp; <?php esc_html_e( 'Developed by', 'devsniper' ); ?> <a href="https://interest.org.il" target="_blank" rel="noopener noreferrer">Interest</a>. <?php esc_html_e( 'All rights reserved.', 'devsniper' ); ?></p>
		</div>
		<?php
	}

	/**
	 * Capture admin menu items into option.
	 */
	public static function capture_admin_menu() {
		if ( ! DevSniper_Security::current_user_can_search() ) {
			return;
		}

		global $menu, $submenu;

		$pages = array();

		if ( is_array( $menu ) ) {
			foreach ( $menu as $m ) {
				if ( empty( $m[0] ) || empty( $m[2] ) ) {
					continue;
				}

				$top_title = wp_strip_all_tags( $m[0] );
				$top_slug  = $m[2];

				if ( false !== stripos( $top_slug, 'devsniper' ) ) {
					continue;
				}

				$top_url = self::normalize_admin_url( $top_slug );

				$icon = '';
				if ( ! empty( $m[6] ) ) {
					$icon = $m[6];
				} elseif ( ! empty( $m[4] ) && false !== strpos( $m[4], 'dashicons-' ) ) {
					$icon = $m[4];
				}

				$pages[] = array(
					'title' => $top_title,
					'path'  => $top_title,
					'url'   => $top_url,
					'icon'  => $icon,
				);

				if ( isset( $submenu[ $top_slug ] ) && is_array( $submenu[ $top_slug ] ) ) {
					foreach ( $submenu[ $top_slug ] as $sub ) {
						if ( empty( $sub[0] ) || empty( $sub[2] ) ) {
							continue;
						}

						$sub_title = wp_strip_all_tags( $sub[0] );
						$sub_slug  = $sub[2];
						$sub_url   = self::normalize_admin_url( $sub_slug );

						$pages[] = array(
							'title' => $sub_title,
							'path'  => $top_title . ' > ' . $sub_title,
							'url'   => $sub_url,
							'icon'  => $icon,
						);
					}
				}
			}
		}

		update_option( self::OPTION_ADMIN_PAGES, $pages, false );
	}

	/**
	 * Normalize admin URL from slug.
	 *
	 * @param string $slug Menu slug.
	 * @return string
	 */
	protected static function normalize_admin_url( $slug ) {
		if ( preg_match( '#^https?://#i', $slug ) ) {
			return $slug;
		}

		if ( '' === $slug || '#' === substr( $slug, 0, 1 ) ) {
			return admin_url();
		}

		if ( false !== strpos( $slug, '.php' ) ) {
			return admin_url( $slug );
		}

		if ( false !== strpos( $slug, '=' ) ) {
			return admin_url( 'admin.php?page=' . $slug );
		}

		return admin_url( 'admin.php?page=' . $slug );
	}

	/**
	 * AJAX search handler.
	 */
	public static function ajax_search() {
		check_ajax_referer( DevSniper_Security::AJAX_NONCE_ACTION, 'nonce' );

		if ( ! DevSniper_Security::current_user_can_search() ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized', 'devsniper' ) ), 403 );
		}

		$settings = self::get_settings();
		if ( empty( $settings['enabled'] ) ) {
			wp_send_json_error( array( 'message' => __( 'Search disabled', 'devsniper' ) ), 400 );
		}

		$term = isset( $_GET['q'] ) ? sanitize_text_field( wp_unslash( $_GET['q'] ) ) : '';

		if ( '' === $term ) {
			wp_send_json_success(
				array(
					'posts'       => array(),
					'users'       => array(),
					'media'       => array(),
					'admin_pages' => array(),
				)
			);
		}

		$response = array(
			'posts'       => ! empty( $settings['enable_posts'] ) ? self::search_posts( $term ) : array(),
			'users'       => ! empty( $settings['enable_users'] ) ? self::search_users( $term ) : array(),
			'media'       => ! empty( $settings['enable_media'] ) ? self::search_media( $term ) : array(),
			'admin_pages' => ! empty( $settings['enable_admin_pages'] ) ? self::search_admin_pages( $term ) : array(),
		);

		wp_send_json_success( $response );
	}

	/**
	 * Toggle favourite search item.
	 */
	public static function ajax_toggle_favorite() {
		check_ajax_referer( DevSniper_Security::AJAX_NONCE_ACTION, 'nonce' );

		if ( ! DevSniper_Security::current_user_can_search() ) {
			wp_send_json_error( array( 'message' => __( 'Unauthorized', 'devsniper' ) ), 403 );
		}

		$item = array(
			'id'        => isset( $_POST['id'] ) ? sanitize_text_field( wp_unslash( $_POST['id'] ) ) : '',
			'title'     => isset( $_POST['title'] ) ? sanitize_text_field( wp_unslash( $_POST['title'] ) ) : '',
			'url'       => isset( $_POST['url'] ) ? esc_url_raw( wp_unslash( $_POST['url'] ) ) : '',
			'thumb'     => isset( $_POST['thumb'] ) ? esc_url_raw( wp_unslash( $_POST['thumb'] ) ) : '',
			'icon_class'=> isset( $_POST['icon_class'] ) ? sanitize_text_field( wp_unslash( $_POST['icon_class'] ) ) : '',
			'type'      => isset( $_POST['type'] ) ? sanitize_text_field( wp_unslash( $_POST['type'] ) ) : '',
			'is_active' => isset( $_POST['is_active'] ) ? (bool) $_POST['is_active'] : false,
		);

		$favs = DevSniper_Favorites::toggle_search_favorite( $item );
		wp_send_json_success( array( 'favorites' => array_values( $favs ) ) );
	}

	/**
	 * Search posts + pages.
	 *
	 * @param string $term Search term.
	 * @return array
	 */
	protected static function search_posts( $term ) {
		$needle = self::normalize_term( $term );

		$query = new WP_Query(
			array(
				's'                   => $term,
				'post_type'           => array( 'post', 'page' ),
				'post_status'         => 'any',
				'posts_per_page'      => 20,
				'ignore_sticky_posts' => true,
			)
		);

		$items = array();

		if ( $query->have_posts() ) {
			foreach ( $query->posts as $post ) {
				$title = get_the_title( $post );
				$hay   = self::normalize_term( $title );

				if ( false === strpos( $hay, $needle ) ) {
					continue;
				}

				$thumb_url = '';
				if ( has_post_thumbnail( $post ) ) {
					$t = get_the_post_thumbnail_url( $post, 'thumbnail' );
					if ( $t ) {
						$thumb_url = esc_url( $t );
					}
				}

				$pto        = get_post_type_object( $post->post_type );
				$type_label = $pto ? $pto->labels->singular_name : ucfirst( $post->post_type );

				$items[] = array(
					'id'       => 'post-' . $post->ID,
					'title'    => $title,
					'type'     => $type_label,
					'edit_url' => get_edit_post_link( $post->ID, 'raw' ),
					'thumb'    => $thumb_url,
				);
			}
		}

		return $items;
	}

	/**
	 * Search users.
	 *
	 * @param string $term Search term.
	 * @return array
	 */
	protected static function search_users( $term ) {
		$user_query = new WP_User_Query(
			array(
				'search'         => '*' . esc_attr( $term ) . '*',
				'search_columns' => array( 'user_login', 'user_email', 'display_name' ),
				'number'         => 10,
			)
		);

		$items = array();

		foreach ( $user_query->get_results() as $user ) {
			$avatar_url = '';
			if ( function_exists( 'get_avatar_url' ) ) {
				$avatar_url = esc_url( get_avatar_url( $user->ID, array( 'size' => 64 ) ) );
			}

			$items[] = array(
				'id'       => 'user-' . $user->ID,
				'title'    => $user->display_name,
				'type'     => __( 'User', 'devsniper' ),
				'edit_url' => get_edit_user_link( $user->ID ),
				'thumb'    => $avatar_url,
			);
		}

		return $items;
	}

	/**
	 * Search media library.
	 *
	 * @param string $term Search term.
	 * @return array
	 */
	protected static function search_media( $term ) {
		$needle = self::normalize_term( $term );

		$query = new WP_Query(
			array(
				's'              => $term,
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'posts_per_page' => 20,
			)
		);

		$items = array();

		foreach ( $query->posts as $post ) {
			$title = get_the_title( $post );
			$hay   = self::normalize_term( $title );

			if ( false === strpos( $hay, $needle ) ) {
				continue;
			}

			$thumb_url = '';
			$thumb     = wp_get_attachment_image_src( $post->ID, 'thumbnail' );
			if ( $thumb && ! empty( $thumb[0] ) ) {
				$thumb_url = esc_url( $thumb[0] );
			}

			$items[] = array(
				'id'       => 'media-' . $post->ID,
				'title'    => $title,
				'type'     => __( 'Media', 'devsniper' ),
				'edit_url' => get_edit_post_link( $post->ID, 'raw' ),
				'thumb'    => $thumb_url,
			);
		}

		return $items;
	}

	/**
	 * Search captured admin pages.
	 *
	 * @param string $term Search term.
	 * @return array
	 */
	protected static function search_admin_pages( $term ) {
		$pages = get_option( self::OPTION_ADMIN_PAGES, array() );
		if ( ! is_array( $pages ) || empty( $pages ) ) {
			return array();
		}

		$items  = array();
		$needle = self::normalize_term( $term );

		foreach ( $pages as $page ) {
			if ( empty( $page['title'] ) || empty( $page['url'] ) ) {
				continue;
			}

			$haystack = $page['title'] . ' ' . ( isset( $page['path'] ) ? $page['path'] : '' );
			$hay      = self::normalize_term( $haystack );

			if ( false === strpos( $hay, $needle ) ) {
				continue;
			}

			$icon       = isset( $page['icon'] ) ? $page['icon'] : '';
			$thumb_url  = '';
			$icon_class = '';

			if ( $icon ) {
				if ( preg_match( '#^https?://#i', $icon ) || 0 === strpos( $icon, 'data:image' ) ) {
					$thumb_url = esc_url( $icon );
				} elseif ( false !== strpos( $icon, 'dashicons-' ) ) {
					$icon_class = 'dashicons ' . $icon;
				}
			}

			$items[] = array(
				'id'         => md5( $page['url'] . $page['title'] ),
				'title'      => isset( $page['path'] ) ? $page['path'] : $page['title'],
				'type'       => __( 'Admin page', 'devsniper' ),
				'edit_url'   => $page['url'],
				'thumb'      => $thumb_url,
				'icon_class' => $icon_class,
			);

			if ( count( $items ) >= 30 ) {
				break;
			}
		}

		return $items;
	}

	/**
	 * Normalize term.
	 *
	 * @param string $str String.
	 * @return string
	 */
	protected static function normalize_term( $str ) {
		$str = (string) $str;
		if ( function_exists( 'remove_accents' ) ) {
			$str = remove_accents( $str );
		}
		$str = function_exists( 'mb_strtolower' ) ? mb_strtolower( $str ) : strtolower( $str );
		return $str;
	}
}
