(function () {
  const data = window.DevSniperInspectorData || {};
  const targetCursor = "url(\"data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='32' height='32' viewBox='0 0 32 32'%3E%3Ccircle cx='16' cy='16' r='4' fill='%2310b981'/%3E%3Ccircle cx='16' cy='16' r='11' fill='none' stroke='%2310b981' stroke-width='2'/%3E%3Cpath d='M16 2v6M16 24v6M2 16h6M24 16h6' stroke='%2310b981' stroke-width='2'/%3E%3C/svg%3E\") 16 16, crosshair";
  if (!window.CSS || !CSS.escape) {
    window.CSS = window.CSS || {};
    CSS.escape = CSS.escape || function (sel) {
      return (sel || '').replace(/[^a-zA-Z0-9_-]/g, '\\$&');
    };
  }

  const i18n = Object.assign(
    {
      tag: 'Tag',
      id: 'ID',
      classes: 'Classes',
      size: 'Size',
      offsets: 'Offsets',
      selector: 'Selector',
      path: 'Path',
      source: 'Source',
      open_editor: 'Open editor',
      favorite: 'Favorite',
      favourites: 'Favorites',
      search_favorites: 'Search favorites…',
      remove: 'Remove',
      copy: 'Copy',
      close: 'Close',
      no_favorites: 'No favorites yet.',
      open: 'Open',
    },
    data.i18n || {}
  );

  let overlay = document.getElementById('devsniper-overlay');
  let outline = document.querySelector('.devsniper-inspector-box');
  let label = document.querySelector('.devsniper-inspector-label');
  let panel = document.querySelector('.devsniper-inspector-info');

  // If overlay does not exist (front-end) create it.
  if (!overlay) {
    overlay = document.createElement('div');
    overlay.id = 'devsniper-overlay';
    overlay.className = 'devsniper-overlay';
    document.body.appendChild(overlay);
  }
  if (!outline) {
    outline = document.createElement('div');
    outline.className = 'devsniper-inspector-box';
    document.body.appendChild(outline);
  }
  if (!label) {
    label = document.createElement('div');
    label.className = 'devsniper-inspector-label';
    document.body.appendChild(label);
  }
  if (!panel) {
    panel = document.createElement('div');
    panel.className = 'devsniper-inspector-info';
    document.body.appendChild(panel);
  }
  if (panel) {
    panel.setAttribute('dir', 'ltr');
    panel.setAttribute('lang', 'en');
  }

  let inspectorOn = !!data.enabled;
  let favourites = Array.isArray(data.favorites) ? data.favorites : [];
  let currentElement = null;
  let currentInfo = null;
  let selectionLocked = false;
  let dragState = { dragging: false, startX: 0, startY: 0, panelX: 0, panelY: 0, hasCustomPosition: false };
  let customCssSelectors = null;
  const pluginCatalog = Array.isArray(data.plugins) ? data.plugins : [];

  function escapeHtml(str) {
    if (str === null || typeof str === 'undefined') return '';
    return String(str).replace(/[&<>"']/g, function (m) {
      return {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#39;',
      }[m];
    });
  }

  function setCursor(active) {
    const root = document.documentElement;
    if (active && root) {
      root.style.setProperty('--devsniper-target-cursor', targetCursor);
    } else if (root) {
      root.style.removeProperty('--devsniper-target-cursor');
    }
    document.documentElement.classList.toggle('devsniper-target-cursor', active);
    document.body.classList.toggle('devsniper-target-cursor', active);
    document.documentElement.classList.toggle('devsniper-lock-links', active);
  }

  function pauseTargetCursor() {
    document.documentElement.classList.remove('devsniper-target-cursor');
    document.body.classList.remove('devsniper-target-cursor');
    document.documentElement.classList.remove('devsniper-lock-links');
  }

  function showBackdrop() {
    if (overlay) {
      overlay.style.display = 'block';
    }
  }

  function hideBackdrop() {
    if (overlay) {
      overlay.style.display = 'none';
    }
  }

  function sendToggle(state) {
    fetch(data.ajax_url, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
      body: new URLSearchParams({
        action: 'devsniper_toggle_inspector',
        nonce: data.nonce,
        enabled: state ? 1 : 0,
      }).toString(),
    });
  }

  function setInspector(state, persist) {
    inspectorOn = !!state;
    selectionLocked = false;
    setCursor(inspectorOn);
    if (!inspectorOn) {
      hideOutline();
      hidePanel();
      currentElement = null;
      hideBackdrop();
      dragState.hasCustomPosition = false;
    }
    if (persist) {
      sendToggle(inspectorOn);
    }
  }

  function hideOutline() {
    outline.style.display = 'none';
    label.style.display = 'none';
  }

  function showOutline(el) {
    if (!el) return;
    const rect = el.getBoundingClientRect();
    const scrollX = window.scrollX || window.pageXOffset;
    const scrollY = window.scrollY || window.pageYOffset;
    const top = rect.top + scrollY;
    const left = rect.left + scrollX;
    const rightEdge = window.innerWidth + scrollX;
    outline.style.display = 'block';
    outline.style.left = left + 'px';
    outline.style.top = top + 'px';
    outline.style.width = rect.width + 'px';
    outline.style.height = rect.height + 'px';

    const tag = (el.tagName || '').toLowerCase();
    const id = el.id ? '#' + el.id : '';
    const cls = el.classList && el.classList.length ? '.' + Array.from(el.classList).join('.') : '';
    const size = Math.round(rect.width) + '×' + Math.round(rect.height);

    label.textContent = tag + id + cls + '  ' + size;
    label.style.display = 'block';
    const labRect = label.getBoundingClientRect();
    let lTop = rect.top + scrollY - labRect.height - 8;
    if (lTop < scrollY) lTop = rect.bottom + scrollY + 8;
    let lLeft = left;
    if (lLeft + labRect.width > rightEdge) lLeft = rightEdge - labRect.width - 8;
    label.style.top = lTop + 'px';
    label.style.left = lLeft + 'px';
  }

  function isUiElement(node) {
    if (!node) return false;
    if (panel && panel.contains(node)) return true;
    if (node === outline || node === label) return true;
    if (overlay && (node === overlay || overlay.contains(node))) return true;
    return false;
  }

  function refineTarget(el, x, y) {
    if (!el) return null;
    const stack = typeof document.elementsFromPoint === 'function' ? document.elementsFromPoint(x, y) : [el];
    const preferred = ['img', 'svg', 'path', 'picture', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'span', 'a', 'button', 'label'];
    let best = null;
    let smallestArea = Infinity;

    for (const node of stack) {
      if (!(node instanceof Element)) continue;
      if (isUiElement(node)) continue;
      if (node.tagName && (node.tagName.toLowerCase() === 'html' || node.tagName.toLowerCase() === 'body')) {
        continue;
      }
      const rect = node.getBoundingClientRect();
      if (rect.width <= 0 || rect.height <= 0) continue;

      const area = rect.width * rect.height;
      if (!best) best = node;
      if (preferred.includes((node.tagName || '').toLowerCase())) {
        return node;
      }
      if (area < smallestArea) {
        smallestArea = area;
        best = node;
      }
    }
    return best;
  }

  function buildSelector(el) {
    if (!el || !el.tagName) return '';
    if (el.id) {
      return '#' + CSS.escape(el.id);
    }
    const tag = el.tagName.toLowerCase();
    if (el.classList && el.classList.length) {
      return tag + '.' + Array.from(el.classList).slice(0, 3).map((c) => CSS.escape(c)).join('.');
    }
    let path = tag;
    let cur = el;
    let depth = 0;
    while (cur && cur.parentElement && depth < 6) {
      const parent = cur.parentElement;
      const siblings = Array.from(parent.children).filter((c) => c.tagName === cur.tagName);
      const idx = siblings.indexOf(cur) + 1;
      path = cur.tagName.toLowerCase() + `:nth-child(${idx}) ` + path;
      cur = parent;
      depth++;
      if (cur.id) {
        path = '#' + CSS.escape(cur.id) + ' ' + path;
        break;
      }
    }
    return path.trim();
  }

  function buildDomPath(el) {
    if (!el) return '';
    const parts = [];
    let cur = el;
    let depth = 0;
    while (cur && cur.nodeType === 1 && depth < 8) {
      const parent = cur.parentElement;
      let part = cur.tagName.toLowerCase();
      if (parent) {
        const idx = Array.from(parent.children).indexOf(cur) + 1;
        part += `:nth-child(${idx})`;
      }
      parts.unshift(part);
      if (part === 'body') break;
      cur = parent;
      depth++;
    }
    return parts.join(' > ');
  }

  function normalizeSourceToken(value) {
    return String(value || '').toLowerCase().replace(/[^a-z0-9]/g, '');
  }

  function collectSourceHints(el) {
    const hints = [];
    let node = el;
    let depth = 0;
    while (node && node.nodeType === 1 && depth < 6) {
      if (node.id) hints.push(node.id);
      if (node.classList && node.classList.length) {
        hints.push(...Array.from(node.classList));
      }
      if (node.getAttributeNames) {
        node.getAttributeNames().forEach((name) => {
          if (name.indexOf('data-') === 0 || name === 'id' || name === 'class') {
            hints.push(name);
            const val = node.getAttribute(name);
            if (val) hints.push(val);
          }
        });
      }
      node = node.parentElement;
      depth++;
    }
    return hints.join(' ');
  }

  function detectPluginSource(el) {
    if (!el || !pluginCatalog.length) return null;
    const hay = collectSourceHints(el);
    if (!hay) return null;
    const hayLower = hay.toLowerCase();
    const hayCompact = normalizeSourceToken(hay);
    let match = null;
    pluginCatalog.forEach((plugin) => {
      const slug = (plugin.slug || '').toLowerCase();
      if (!slug || slug === 'devsniper') return;
      const compactSlug = normalizeSourceToken(slug);
      const name = (plugin.name || '').toLowerCase();
      const compactName = normalizeSourceToken(name);
      if (hayLower.indexOf(slug) !== -1 || (compactSlug && hayCompact.indexOf(compactSlug) !== -1)) {
        match = {
          slug: plugin.slug,
          name: plugin.name || plugin.slug,
        };
      }
      if (!match && name && (hayLower.indexOf(name) !== -1 || (compactName && hayCompact.indexOf(compactName) !== -1))) {
        match = {
          slug: plugin.slug,
          name: plugin.name || plugin.slug,
        };
      }
    });
    return match;
  }

  function getCustomCssSelectors() {
    if (Array.isArray(customCssSelectors)) return customCssSelectors;
    customCssSelectors = [];
    const styleNode = document.getElementById('wp-custom-css');
    if (!styleNode || !styleNode.sheet) return customCssSelectors;
    try {
      const rules = styleNode.sheet.cssRules || styleNode.sheet.rules || [];
      for (const rule of rules) {
        if (rule && rule.selectorText) {
          customCssSelectors.push(rule.selectorText);
        }
      }
    } catch (e) {
      return customCssSelectors;
    }
    return customCssSelectors;
  }

  function matchesCustomCss(el) {
    if (!el || typeof el.matches !== 'function') return false;
    const selectors = getCustomCssSelectors();
    if (!selectors.length) return false;
    for (const selector of selectors) {
      try {
        if (selector && el.matches(selector)) return true;
      } catch (e) {
        continue;
      }
    }
    return false;
  }

  function detectSourceDetails(el) {
    if (!el) return { label: 'Theme/Unknown', type: 'theme', plugin: null };
    if (isUiElement(el)) return { label: 'DevSniper', type: 'devsniper', plugin: null };

    const inWpToolbar = el.closest && el.closest('#wpadminbar');
    if (inWpToolbar) return { label: 'WP Toolbar', type: 'wp_toolbar', plugin: null };

    const isWoo = el.closest && el.closest('.woocommerce, [class*="wc-"], form.cart, .woocommerce-page');
    if (isWoo) return { label: 'WooCommerce', type: 'woocommerce', plugin: { slug: 'woocommerce', name: 'WooCommerce' } };

    const elementorElement = el.closest && el.closest('.elementor-element[data-id]');
    if (elementorElement && elementorElement !== document.body) {
      return { label: 'Elementor', type: 'elementor', plugin: { slug: 'elementor', name: 'Elementor' } };
    }

    const gutenbergAncestor = el.closest && el.closest('.wp-block, [class*=\"wp-block-\"], .block-editor-block-list__layout, [data-type^=\"core/\"]');
    if (gutenbergAncestor && gutenbergAncestor !== document.body) return { label: 'Gutenberg', type: 'gutenberg', plugin: null };

    const templatePart = el.closest && el.closest('header, footer, .site-header, .site-footer, .wp-block-template-part, [data-template-part]');
    if (templatePart) return { label: 'Template', type: 'template', plugin: null };

    if (matchesCustomCss(el)) return { label: 'Custom CSS', type: 'custom_css', plugin: null };

    const pluginMatch = detectPluginSource(el);
    if (pluginMatch) return { label: 'Plugin: ' + pluginMatch.name, type: 'plugin', plugin: pluginMatch };

    return { label: 'Theme/Unknown', type: 'theme', plugin: null };
  }

  function getOffsets(el) {
    const rect = el.getBoundingClientRect();
    return {
      width: Math.round(rect.width) + 'px',
      height: Math.round(rect.height) + 'px',
      top: Math.round(rect.top + window.scrollY) + 'px',
      left: Math.round(rect.left + window.scrollX) + 'px',
    };
  }

  function findAnchor(el) {
    if (!el) return '';
    if (el.id) return '#' + el.id;
    const withId = el.closest && el.closest('[id]');
    if (withId && withId.id) return '#' + withId.id;
    return '';
  }

  function findElementorId(el) {
    const elementorEl = el && el.closest ? el.closest('.elementor-element[data-id]') : null;
    return elementorEl && elementorEl.dataset && elementorEl.dataset.id ? elementorEl.dataset.id : '';
  }

  function findElementorDocumentId(el) {
    if (!el || !el.closest) return '';
    const doc = el.closest('[data-elementor-id]');
    if (!doc) return '';
    const raw = doc.getAttribute('data-elementor-id') || '';
    const cleaned = String(raw).replace(/[^0-9]/g, '');
    return cleaned || '';
  }

  function extractPreview(el) {
    if (!el) return null;
    const lowerTag = (el.tagName || '').toLowerCase();
    let src = '';
    if (lowerTag === 'img' && el.src) {
      src = el.currentSrc || el.src;
    }
    if (!src) {
      const img = el.querySelector && el.querySelector('img');
      if (img && img.src) {
        src = img.currentSrc || img.src;
      }
    }
    if (!src) {
      const bg = getComputedStyle(el).backgroundImage;
      const match = bg && bg !== 'none' ? bg.match(/url\\(["']?(.*?)["']?\\)/) : null;
      if (match && match[1]) {
        src = match[1];
      }
    }
    if (!src) return null;
    return {
      src: src,
      alt: (el.getAttribute && el.getAttribute('alt')) || lowerTag || 'element',
    };
  }

  function copy(val, btn) {
    if (!val) return;
    navigator.clipboard &&
      navigator.clipboard.writeText(val).then(() => {
        if (btn) {
          btn.setAttribute('data-copied-label', i18n.copied || 'Copied');
          btn.classList.add('is-copied');
          setTimeout(() => btn.classList.remove('is-copied'), 1000);
        }
      });
  }

  function hidePanel() {
    panel.style.display = 'none';
    panel.innerHTML = '';
  }

  function ensureFavoritePreview(fav) {
    if (!fav) return null;
    if (fav.preview) return fav.preview;
    if (fav.path || fav.selector) {
      const el = document.querySelector(fav.path || fav.selector);
      const p = extractPreview(el);
      if (p) {
        fav.preview = p;
      }
    }
    return fav.preview || null;
  }

  function renderFavourites(filter) {
    const list = panel.querySelector('.devs-fav-list');
    if (!list) return;
    list.innerHTML = '';
    const items = favourites.filter((item) => {
      if (!filter) return true;
      const hay = (item.title || '') + ' ' + (item.selector || '') + ' ' + (item.source || '');
      return hay.toLowerCase().indexOf(filter.toLowerCase()) !== -1;
    });
    if (!items.length) {
      list.innerHTML = '<li class="devs-fav-empty">' + escapeHtml(i18n.no_favorites) + '</li>';
      return;
    }
    items.forEach((item) => {
      const li = document.createElement('li');
      li.className = 'devs-fav-item';

      const thumb = document.createElement('div');
      thumb.className = 'devs-fav-thumb';
      const preview = ensureFavoritePreview(item);
      if (preview && preview.src) {
        const img = document.createElement('img');
        img.src = preview.src;
        img.alt = preview.alt || '';
        thumb.appendChild(img);
      } else {
        const fallback = document.createElement('span');
        const title = (item.title || item.selector || '?').trim();
        fallback.textContent = title ? title.charAt(0).toUpperCase() : '?';
        thumb.appendChild(fallback);
      }
      li.appendChild(thumb);

      const content = document.createElement('div');
      content.className = 'devs-fav-content';

      const titleEl = document.createElement('div');
      titleEl.className = 'devs-fav-title';
      const star = document.createElement('span');
      star.className = 'devs-fav-star';
      star.textContent = '★';
      titleEl.appendChild(star);
      const titleText = document.createElement('span');
      titleText.textContent = item.title || item.selector || '';
      titleEl.appendChild(titleText);
      content.appendChild(titleEl);

      const meta = document.createElement('div');
      meta.className = 'devs-fav-meta';
      meta.textContent = item.selector || '';
      content.appendChild(meta);
      li.appendChild(content);

      const actions = document.createElement('div');
      actions.className = 'devs-fav-actions';
      const open = document.createElement('a');
      open.href = item.url || '';
      open.target = '_blank';
      open.rel = 'noopener noreferrer';
      open.textContent = i18n.open;
      actions.appendChild(open);
      const removeBtn = document.createElement('button');
      removeBtn.type = 'button';
      removeBtn.className = 'devs-btn-link devs-fav-remove';
      removeBtn.dataset.remove = item.id || '';
      removeBtn.textContent = i18n.remove;
      actions.appendChild(removeBtn);
      li.appendChild(actions);

      list.appendChild(li);
    });
  }

  function keepPanelInView() {
    if (!panel) return;
    const rect = panel.getBoundingClientRect();
    let left = rect.left;
    let top = rect.top;
    const maxLeft = Math.max(8, window.innerWidth - rect.width - 8);
    const maxTop = Math.max(8, window.innerHeight - rect.height - 8);
    left = Math.min(Math.max(8, left), maxLeft);
    top = Math.min(Math.max(8, top), maxTop);
    panel.style.left = left + 'px';
    panel.style.top = top + 'px';
    panel.style.right = 'auto';
    panel.style.bottom = 'auto';
  }

  let panelResizeObserver = null;
  function ensurePanelResizeObserver() {
    if (panelResizeObserver || !panel || typeof ResizeObserver === 'undefined') {
      return;
    }

    panelResizeObserver = new ResizeObserver(function () {
      if (panel && panel.style.display !== 'none') {
        keepPanelInView();
      }
    });

    panelResizeObserver.observe(panel);
  }

  function row(labelText, value, copyVal, secondary, extraClass) {
    const labelHtml = '<div class="devs-label">' + escapeHtml(labelText || '') + '</div>';
    const valHtml = '<div class="devs-value-main">' + escapeHtml(value || '') + (secondary ? ' · ' + escapeHtml(secondary) : '') + '</div>';
    const copyBtn = copyVal ? '<button class="devs-btn-link devs-copy" data-copy="' + escapeHtml(copyVal) + '">' + escapeHtml(i18n.copy) + '</button>' : '';
    const cls = extraClass ? 'devs-row ' + extraClass : 'devs-row';
    return '<div class="' + cls + '">' + labelHtml + '<div class="devs-value">' + valHtml + copyBtn + '</div></div>';
  }

  function positionPanelDefault() {
    const width = panel.offsetWidth || 360;
    const height = panel.offsetHeight || 0;
    const left = Math.max(8, window.innerWidth - width - 16);
    const top = Math.max(8, window.innerHeight - height - 16);
    panel.style.left = left + 'px';
    panel.style.top = top + 'px';
    panel.style.right = 'auto';
    panel.style.bottom = 'auto';
    dragState.panelX = left;
    dragState.panelY = top;
  }

  function renderPanel(el) {
    if (!el) return;
    const selector = buildSelector(el);
    const domPath = buildDomPath(el);
    const sourceDetails = detectSourceDetails(el);
    const source = sourceDetails.label;
    const offsets = getOffsets(el);
    const classes = el.classList && el.classList.length ? '.' + Array.from(el.classList).join('.') : '';
    const title = (el.tagName || 'element').toLowerCase() + (el.id ? '#' + el.id : '');
    const preview = extractPreview(el);
    const anchor = findAnchor(el);
    const elementorId = findElementorId(el);
    const elementorDocumentId = findElementorDocumentId(el);

    currentInfo = {
      title: title,
      selector: selector,
      path: domPath,
      source: source,
      sourceType: sourceDetails.type,
      pluginSlug: sourceDetails.plugin ? sourceDetails.plugin.slug : '',
      url: data.page_url || '',
      preview: preview,
      anchor: anchor,
      elementorId: elementorId,
      elementorDocumentId: elementorDocumentId,
    };

    const header =
      '<div class="devs-header">' +
      '<div class="devs-title">DevSniper Inspector <span class="devs-title-icon">🎯</span></div>' +
      '<div class="devs-actions">' +
      '<button class="devs-close-btn" aria-label="' + escapeHtml(i18n.close) + '">×</button>' +
      '</div>' +
      '</div>' +
      '<div class="devs-subtitle">Hover to highlight, click to inspect. Press Esc to exit.</div>' +
      '<div class="devs-cta">' +
      '<button class="devs-btn devs-btn-ghost devs-open-editor" type="button">' + escapeHtml(i18n.open_editor) + '</button>' +
      '<button class="devs-btn devs-btn-primary devs-add-fav" type="button">' + escapeHtml(i18n.favorite) + '</button>' +
      '</div>';

    const rows = [];
    rows.push(row(i18n.tag, (el.tagName || '').toLowerCase(), null));
    if (el.id) {
      rows.push(row(i18n.id, '#' + el.id, '#' + el.id));
    }
    if (classes) {
      rows.push(row(i18n.classes, classes, classes));
    }
    const sizeDisplay = 'W ' + offsets.width + ' 🔗 H ' + offsets.height;
    const offsetsDisplay = offsets.top + ' · ' + offsets.left;
    rows.push(row(i18n.size, sizeDisplay + ' | ' + offsetsDisplay, null));
    rows.push(row(i18n.selector, selector, selector));
    rows.push(row(i18n.path, domPath, domPath));
    rows.push(row(i18n.source, source, null, null, 'is-source'));

    const favBlock =
      '<div class="devs-fav-block">' +
      '<div class="devs-fav-head">' + escapeHtml(i18n.favourites) + '</div>' +
      '<input type="search" class="devs-fav-search" placeholder="' + escapeHtml(i18n.search_favorites) + '">' +
      '<ul class="devs-fav-list"></ul>' +
      '</div>';

    panel.innerHTML = header + '<div class="devs-rows">' + rows.join('') + '</div>' + favBlock;
    panel.style.display = 'block';
    showBackdrop();
    ensurePanelResizeObserver();

    if (!dragState.hasCustomPosition) {
      positionPanelDefault();
    }
    keepPanelInView();

    renderFavourites('');
    setTimeout(keepPanelInView, 0);
  }

  function addFavourite() {
    if (!currentInfo || !currentInfo.selector) return;
    fetch(data.ajax_url, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
      body: new URLSearchParams({
        action: 'devsniper_save_element',
        nonce: data.nonce,
        title: currentInfo.title || '',
        selector: currentInfo.selector,
        path: currentInfo.path || '',
        source: currentInfo.source || '',
        url: currentInfo.url || '',
      }).toString(),
    })
      .then((r) => r.json())
      .then((json) => {
        if (json && json.success && json.data && Array.isArray(json.data.favorites)) {
          favourites = json.data.favorites.map((fav) => {
            if (
              currentInfo &&
              currentInfo.selector &&
              fav.selector === currentInfo.selector &&
              (!currentInfo.path || fav.path === currentInfo.path) &&
              currentInfo.preview
            ) {
              return Object.assign({}, fav, { preview: currentInfo.preview });
            }
            return fav;
          });
          const favBtn = panel.querySelector('.devs-add-fav');
          if (favBtn) {
            favBtn.innerHTML = '★ ' + escapeHtml(i18n.favorite);
            favBtn.classList.add('is-saved');
          }
          renderFavourites('');
          finishAction({ refresh: true });
        }
      });
  }

  function removeFavourite(id) {
    if (!id) return;
    fetch(data.ajax_url, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
      body: new URLSearchParams({
        action: 'devsniper_delete_element',
        nonce: data.nonce,
        id: id,
      }).toString(),
    })
      .then((r) => r.json())
      .then((json) => {
        if (json && json.success && json.data && Array.isArray(json.data.favorites)) {
          favourites = json.data.favorites;
          renderFavourites('');
          finishAction({ close: true });
        }
      });
  }

  function flushCaches() {
    return fetch(data.ajax_url, {
      method: 'POST',
      credentials: 'same-origin',
      headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' },
      body: new URLSearchParams({
        action: 'devsniper_flush_caches',
        nonce: data.nonce,
      }).toString(),
    });
  }

  function finishAction(options) {
    const opts = options || {};
    if (opts.refresh) {
      flushCaches()
        .catch(() => {})
        .finally(() => {
          setInspector(false, true);
          window.location.reload();
        });
      return;
    }
    if (opts.close) {
      setInspector(false, true);
    }
  }

  function openEditor() {
    if (currentInfo && currentInfo.source === 'Elementor' && data.elementor_url) {
      try {
        const url = new URL(data.elementor_url, window.location.href);
        if (currentInfo.elementorDocumentId) {
          url.searchParams.set('post', currentInfo.elementorDocumentId);
        }
        if (currentInfo.elementorId) {
          url.searchParams.set('devsniper_select', currentInfo.elementorId);
        }
        window.open(url.toString(), '_blank');
      } catch (e) {
        const postSuffix = currentInfo.elementorDocumentId ? '&post=' + encodeURIComponent(currentInfo.elementorDocumentId) : '';
        const selectSuffix = currentInfo.elementorId ? '&devsniper_select=' + encodeURIComponent(currentInfo.elementorId) : '';
        window.open(data.elementor_url + postSuffix + selectSuffix, '_blank');
      }
      finishAction({ close: true });
      return;
    }

    const editorUrls = data.editor_urls || {};
    if (currentInfo && currentInfo.sourceType === 'custom_css' && editorUrls.custom_css) {
      window.open(editorUrls.custom_css, '_blank');
      finishAction({ close: true });
      return;
    }

    if (currentInfo && currentInfo.sourceType === 'template') {
      const templateUrl = editorUrls.site_editor || editorUrls.theme_editor;
      if (templateUrl) {
        window.open(templateUrl, '_blank');
        finishAction({ close: true });
        return;
      }
    }

    if (currentInfo && currentInfo.sourceType === 'plugin' && currentInfo.pluginSlug) {
      const pluginUrls = editorUrls.plugins || {};
      const pluginUrl = pluginUrls[currentInfo.pluginSlug];
      if (pluginUrl) {
        window.open(pluginUrl, '_blank');
        finishAction({ close: true });
        return;
      }
    }

    if (editorUrls.post) {
      const anchor = currentInfo && currentInfo.anchor ? currentInfo.anchor : '';
      try {
        const url = new URL(editorUrls.post, window.location.href);
        if (anchor) {
          url.hash = anchor;
        }
        window.open(url.toString(), '_blank');
      } catch (e) {
        window.open(editorUrls.post + (anchor || ''), '_blank');
      }
      finishAction({ close: true });
      return;
    }

    if (editorUrls.site_editor) {
      window.open(editorUrls.site_editor, '_blank');
      finishAction({ close: true });
      return;
    }

    if (editorUrls.theme_editor) {
      window.open(editorUrls.theme_editor, '_blank');
      finishAction({ close: true });
      return;
    }

    if (editorUrls.custom_css) {
      window.open(editorUrls.custom_css, '_blank');
      finishAction({ close: true });
    }
  }

  // Events
  document.addEventListener('mousemove', function (e) {
    if (!inspectorOn || selectionLocked) return;
    const target = refineTarget(document.elementFromPoint(e.clientX, e.clientY), e.clientX, e.clientY);
    if (!target) return;
    currentElement = target;
    showOutline(target);
  }, true);

  document.addEventListener('click', function (e) {
    const bar = e.target.closest('#wp-admin-bar-devsniper-search a');
    if (bar) {
      e.preventDefault();
      setInspector(!inspectorOn, true);
      return;
    }

    if (!inspectorOn) return;

    if (overlay && overlay.contains(e.target)) {
      setInspector(false, true);
      return;
    }

    const insidePanel = panel.contains(e.target);
    if (insidePanel) return;
    if (selectionLocked) return;

    const target = refineTarget(document.elementFromPoint(e.clientX, e.clientY), e.clientX, e.clientY);
    if (!target) return;
    e.preventDefault();
    e.stopPropagation();
    currentElement = target;
    renderPanel(target);
    selectionLocked = true;
    setCursor(false);
  }, true);

  document.addEventListener('keydown', function (e) {
    const key = (e.key || '').toLowerCase();
    const isMac = navigator.platform.toUpperCase().indexOf('MAC') >= 0;
    if ((isMac && e.metaKey && key === 'k') || (!isMac && e.ctrlKey && key === 'k')) {
      e.preventDefault();
      setInspector(!inspectorOn, true);
    }
    if (key === 'escape' && inspectorOn) {
      e.preventDefault();
      setInspector(false, true);
    }
  });

  overlay.addEventListener('click', function () {
    if (panel.style.display !== 'none') {
      setInspector(false, true);
    }
  });

  panel.addEventListener('click', function (e) {
    if (e.target.classList.contains('devs-copy')) {
      copy(e.target.dataset.copy, e.target);
    }
    if (e.target.classList.contains('devs-close-btn')) {
      setInspector(false, true);
    }
    if (e.target.classList.contains('devs-open-editor')) {
      openEditor();
    }
    if (e.target.classList.contains('devs-add-fav')) {
      addFavourite();
    }
    if (e.target.classList.contains('devs-fav-remove')) {
      removeFavourite(e.target.dataset.remove);
    }
    if (e.target.tagName === 'A' && e.target.closest('.devs-fav-actions')) {
      finishAction({ close: true });
    }
  });

  panel.addEventListener('input', function (e) {
    if (e.target.classList.contains('devs-fav-search')) {
      renderFavourites(e.target.value || '');
    }
  });

  panel.addEventListener('mouseenter', function () {
    pauseTargetCursor();
  });

  panel.addEventListener('mouseleave', function () {
    setCursor(inspectorOn && !selectionLocked);
  });

  // Dragging support (grab header)
  panel.addEventListener('mousedown', function (e) {
    if (!e.target.closest('.devs-header')) return;
    dragState.dragging = true;
    panel.classList.add('is-dragging');
    dragState.startX = e.clientX;
    dragState.startY = e.clientY;
    const rect = panel.getBoundingClientRect();
    dragState.panelX = rect.left;
    dragState.panelY = rect.top;
    panel.style.right = 'auto';
    panel.style.bottom = 'auto';
    e.preventDefault();
  });

  document.addEventListener('mousemove', function (e) {
    if (!dragState.dragging) return;
    const dx = e.clientX - dragState.startX;
    const dy = e.clientY - dragState.startY;
    const rect = panel.getBoundingClientRect();
    let nx = dragState.panelX + dx;
    let ny = dragState.panelY + dy;
    nx = Math.max(8, Math.min(nx, window.innerWidth - rect.width - 8));
    ny = Math.max(8, Math.min(ny, window.innerHeight - rect.height - 8));
    panel.style.left = nx + 'px';
    panel.style.top = ny + 'px';
  });

  document.addEventListener('mouseup', function () {
    if (dragState.dragging) {
      dragState.dragging = false;
      dragState.hasCustomPosition = true;
      panel.classList.remove('is-dragging');
      keepPanelInView();
    }
  });

  window.addEventListener('resize', keepPanelInView);

  setInspector(inspectorOn, false);
})();
